package inc.monsters.mias.data;

import java.io.Serializable;
import java.util.Calendar;
import java.util.Date;

@SuppressWarnings("serial")
public class Kid implements Serializable {
	private String firstName;
	private String lastName;
	private Date birthDate;
	private Date lastScared;
	private double braveness;
	private String country;
	private String photo;

	/**
	 * Constructs a new Kid object. 
	 * 
	 * @param firstName first name of the Kid
	 * @param lastName last name of the Kid
	 * @param birthDate birth date of the Kid
	 * @param braveness braveness factor of the Kid, see {{@link #setBraveness(double)}
	 * @param country the country the Kid lives in
	 * @param lastScared the date this Kid was scared for the last time
	 */
	public Kid(String firstName, String lastName, Date birthDate,
			   double braveness, String country, Date lastScared) {
		super();
		this.firstName = firstName;
		this.lastName = lastName;
		this.birthDate = birthDate;
		this.braveness = braveness;
		this.country = country;
		this.lastScared = lastScared;
	}

	/**
	 * Returns the first name of the kid. 
	 * 
	 * @return the first name of this Kid.
	 */
	public String getFirstName() {
		return firstName;
	}
	
	/**
	 * Set the first name of the kid.
	 * 
	 * @param firstName the first name of the kid.
	 */
	public void setFirstName(String firstName) {
		this.firstName = firstName;
	}

	/**
	 * Returns the last name of the kid. 
	 * 
	 * @return the last name of this Kid.
	 */
	public String getLastName() {
		return lastName;
	}
	
	/**
	 * Set the last name of the kid.
	 * @param lastName the last name of the kid.
	 */
	public void setLastName(String lastName) {
		this.lastName = lastName;
	}

	/**
	 * Gets the birth date of this kid. 
	 * 
	 * @see Kid#getAge()
	 * @return the birth date
	 */
	public Date getBirthDate() {
		return birthDate;
	}
	
	/**
	 * Set the birth date of the kid.
	 * @param birthDate the birth date of the kid
	 */
	public void setBirthDate(Date birthDate) {
		this.birthDate = birthDate;
	}

	/**
	 * Returns the age (in years) of the Kid. This value is calculated from
	 * the birth date, hence there is no setter for this value.
	 * 
	 * @see Kid#getBirthDate()
	 * @return the age of the Kid.
	 */
	public int getAge() {
		Calendar today = Calendar.getInstance();
		Calendar birth = Calendar.getInstance();
		birth.setTime(getBirthDate());
		
		int age = today.get(Calendar.YEAR) - birth.get(Calendar.YEAR);
		if (today.get(Calendar.DAY_OF_YEAR) < birth.get(Calendar.DAY_OF_YEAR)) {
			age--;
		}
		
		return age;
	}

	/**
	 * Returns the date on which this Kid was scared for the last time.
	 * 
	 * @return the date this Kid was scared for the last time
	 */
	public Date getLastScared() {
		return lastScared;
	}

	/**
	 * Set the date when this Kid was scared the last time.
	 * 
	 * @param lastScared the date this Kid was scared for the last time.
	 */
	public void setLastScared(Date lastScared) {
		this.lastScared = lastScared;
	}

	/**
	 * Returns the braveness of the kid on a scale from 0.0 to 10.0
	 * 
	 * @see Kid#setBraveness(double)
	 * @return the braveness of the kid
	 */
	public double getBraveness() {
		return braveness;
	}

	/**
	 * Sets the braveness factor of this Kid. Braveness should be a value 
	 * between 0.0 and 10.0, where larger means braver. A brave Kid is 
	 * harder to scare than a not-so-brave Kid. It is assumed that if the
	 * braveness of the Kid is updated, the Kid was scared today, so the
	 * lastScared date will be set to today automatically.
	 * 
	 * @param braveness the braveness of the Kid on a scale from 0.0 to 10.0
	 */
	public void setBraveness(double braveness) {
		if (braveness > 0.0 && braveness <= 10.0) {
			this.braveness = braveness;
			setLastScared(new Date());
		}
	}

	/**
	 * Get the name of the country where this Kid lives.
	 * 
	 * @return the name of the country where this Kid lives
	 */
	public String getCountry() {
		return country;
	}

	/**
	 * Set the name of the country where this Kid lives. 
	 * 
	 * @param country the name of the country where this Kid lives
	 */
	public void setCountry(String country) {
		this.country = country;
	}

	/**
	 * Get the file name of the photo of this kid. Returns null
	 * if there is no photo.
	 * 
	 * @return the file name of the photo of this kid
	 */
	public String getPhoto() {
		return photo;
	}

	/**
	 * Set the file name of the photo of the kid.
	 * 
	 * @param photo the file name of the photo of the kid.
	 */
	public void setPhoto(String photo) {
		this.photo = photo;
	}


}
